*********************************************************************************;
* Name: All Cause 30-Day Readmission Measure Using Medicare Claims Data         *;
* Company: YALE/YNHH CORE                                                       *;
* Date: last revised on 08/17/2018                                              *;
* Platform: SAS 9.3 or greater                                                  *;
* Note:                                                                         *;
*       1.Planned Readmission Version 4.0 added for 2016 Public Reporting       *;
*       2.Input data must conform to the specifications of R&A                  *;
*********************************************************************************;
OPTIONS SYMBOLGEN MPRINT; 
*****************************************************************************;
* Specify year, disease condition, and various file paths                   *;
*****************************************************************************;

%LET YEAR=1516;   
%LET CONDITION=ACR; 
%LET RawData=  ; *Path for raw data*;
%LET attrData= ; *Path for attribution data;
%LET ccMapData=; *Path for cc map data;
%LET OutData=  ; *Path for derived data sets*;
%LET Macro=    ; *Path for SAS macros*;
%LET FMTS=     ; *Path for CC and CCS ICD0 Formats*;

LIBNAME RAW  "&RawData"   access=readonly;    
LIBNAME ATTR "&attrData"  access=readonly;
libname CC   "&ccMapData" access=readonly;
LIBNAME R    "&OutData";      
LIBNAME F    "&FMTS"      access=readonly;      

OPTIONS FMTSEARCH=(F) SYMBOLGEN MPRINT compress=yes;

%INCLUDE "&Macro.\all_cause_readmission_macros_PY2017.sas";    

%LET ADMISSION=   RAW.index_hw_&year.; 
%LET POST=        RAW.postindex_hw_&year.; 
%LET HXDIAG=      RAW.diaghistory_hw_&year;
%LET TIN_FILE=    attr.attribution_file_&year;
%LET V22_ICD_HCC= CC.dxicdhcc; * File with HCC for V22;

%LET ALL=R.&CONDITION._READM_ALL;
%LET ANALYSIS=R.&CONDITION._READM_ANALYSIS;
%LET ACR_RSRR=R.&CONDITION._READM_RSRR;
%LET RESULTS=R.&CONDITION._RSRR_BS; 
%LET EST=R.&CONDITION._READM_EST;

%LET MODEL_VAR=AGE_65 MetaCancer SevereCancer OtherCancer Hematological Coagulopathy 
	IronDeficiency LiverDisease PancreaticDisease OnDialysis RenalFailure Transplants
	HxInfection OtherInfectious Septicemia CHF CADCVD Arrhythmias CardioRespiratory 
	COPD LungDisorder Malnutrition MetabolicDisoder Arthritis Diabetes Ulcers
	MotorDisfunction Seizure RespiratorDependence  Alcohol Psychological HipFracture;   

%LET TRANS_DIAG=R.DIAGHISTORY_&CONDITION._&YEAR._TRANS;

* rename variables from admission, create trans_diag, then limit index cases having PARA=1;
data index ;
    set &admission;
    rename admsn_dt = admit
           dschrgdt = disch
           edgnsd01 = edgscd01
           edgnsd02 = edgscd02
           edgnsd03 = edgscd03
           edgnsd04 = edgscd04
           edgnsd05 = edgscd05
           edgnsd06 = edgscd06
           edgnsd07 = edgscd07
           edgnsd08 = edgscd08
           edgnsd09 = edgscd09
           edgnsd10 = edgscd10
           edgnsd11 = edgscd11
           edgnsd12 = edgscd12
            ;
    keep bid_mpr_3 birth death admsn_dt  dschrgdt edgnsd: provider
    case ddest diag1-diag25 group history_case para postmod_a post_flag
    premo_a proc1-proc25 pstate sex trans trans_first trans_last trans_mid
    txflag year stayid CLM_DGNS_PRCDR_ICD_IND;
run ;
%RETRIEVE_HX(index,&TRANS_DIAG);  
data index;
    set index(where=(PARA=1));
run;

* rename post-index variables;
data postindex ;
    set &post ;
    keep admdiff admsn_dt case
    ddest diag1-diag25 dschrgdt
    edgnsd:
    proc1-proc25
    procdt1-procdt25
    provider bid_mpr_3
    CLM_DGNS_PRCDR_ICD_IND
    ;
    rename admsn_dt = admit
           dschrgdt = disch
           edgnsd01 = edgscd01
           edgnsd02 = edgscd02
           edgnsd03 = edgscd03
           edgnsd04 = edgscd04
           edgnsd05 = edgscd05
           edgnsd06 = edgscd06
           edgnsd07 = edgscd07
           edgnsd08 = edgscd08
           edgnsd09 = edgscd09
           edgnsd10 = edgscd10
           edgnsd11 = edgscd11
           edgnsd12 = edgscd12
            ;
run ;

/* ELIMINATE ADMISSIONS THAT APPEAR TWICE (ACROSS YEARS)*/
PROC SORT DATA=INDEX NODUPKEY DUPOUT=QA_DupOut EQUALS;
    BY bid_mpr_3 ADMIT DISCH provider;
RUN;

/*** merge TIN_ASSIGNED  ***/
data tin_data;
   set &tin_file (keep = bid_mpr_3 tin_assigned);
run;
proc sort data=tin_data;
    by bid_mpr_3;
run;
data index;
    merge   index (in=a) tin_data (in=b);
    by bid_mpr_3;
    if a and b;
run;

/* IDENTIFY AND COMBINE TWO ADJACENT AMI ADMISSIONS (disch1=admit2), USE DISCHARGE DATE
    OF 2ND ADMISSION TO REPLACE DISCHARGE DATE OF 1ST ADMISSION (disch1=disch2),
    SAME FOR DISCHARGE STATUS, TRANS_FIRST, TRANS_MID, POSTMOD.
    ALSO, CREATE CASE_P TO BE USED FOR FINDING READMISSION.
    THIS WORKS WHEN THERE ARE MORE THAN TWO ADJACENT AMI ADMISSIONS. */

DATA TEMP;
    SET INDEX;
    BY bid_mpr_3;   
if (admit <= lag(disch) <= disch) and lag(provider)=provider
    and lag(bid_mpr_3)=bid_mpr_3 and lag(diag1) = diag1
     then combine0=1;
else combine0=0;
RUN;

proc sort data=TEMP;
    by bid_mpr_3 descending admit descending disch;
run;

data TEMP2 QA_CombOut_mid;
set TEMP;
by bid_mpr_3;

if (admit <= lag(admit) <= disch) and
    lag(provider)=provider
    and lag(bid_mpr_3)=bid_mpr_3 and lag(diag1) = diag1
    then combine=1;
else combine=0;
if combine0 and combine then output QA_CombOut_mid;
else output TEMP2;

run;

data TEMP3 QA_CombOut_last;
set TEMP2;

disch_2=lag(disch);
case_2=lag(case);
ddest_2=lag(ddest);
trans_first_2=lag(trans_first);
trans_mid_2=lag(trans_mid);
postmod_a2=lag(postmod_a);
if lag(provider)=provider and lag(bid_mpr_3)=bid_mpr_3 and lag(combine0)=1 then do;
    disch=disch_2;
    case_p=case_2;
    ddest=ddest_2;
    trans_first=trans_first_2;
    trans_mid=trans_mid_2;
    postmod_a=postmod_a2;
    end;
else case_p=case;

drop disch_2 case_2 ddest_2 trans_first_2 trans_mid_2 postmod_a2;

if combine0 ^=1 then output TEMP3;
else output QA_CombOut_last;

run;

PROC SORT DATA=TEMP3;
    BY bid_mpr_3 DESCENDING ADMIT  DESCENDING DISCH PROVIDER;
RUN;


/*  Create study cohort
    by APPLYing THE FOLLOWING INCLUSION AND EXCLUSION CRITERIA:
      AGE >=65,   DEAD=0, PREMO=12, POSTMOD=1, 2, 3,
    TRANS_COMBINE=0, AMA=0 */

DATA ALL;
    SET TEMP3 (DROP=COMBINE0);
    BY bid_mpr_3;
ATTRIB TRANSFER_OUT LABEL='TRANSFER OUT' LENGTH=3.;
ATTRIB TRANS_COMBINE LABEL='TRANSFER OUT' LENGTH=3.;
ATTRIB DD30 LABEL='30-DAY MORTALITY FROM DISCHARGE' LENGTH=3.;
ATTRIB AGE_65 LABEL='YEARS OVER 65' LENGTH=3.;
ATTRIB AGE65 LABEL='AGE GE 65' LENGTH=3.;
ATTRIB DEAD LABEL='IN HOSPITAL DEATH' LENGTH=3.;
ATTRIB sample LABEL='MEET INC & EXL CRITERIA' LENGTH=3.;
* 12/14/2017: non-PRA update, change length or proccc variables;
 LENGTH ADDXG $6. DCGDIAG $7. DCGVRS $1. proccc1-proccc25 $7. rehabexcl 3.;/* AP?PS added version and changed length to 7 char*/
 DCGDIAG = diag1;
 DCGVRS  = CLM_DGNS_PRCDR_ICD_IND; /* added PVP version */

 /* AP/PS - Checking diag version*/
 if DCGVRS = "9" then
  /*ADDXG = PUT(DCGDIAG,$V9CCS.);*/
 put "ERROR: has ICD9";
 else
  ADDXG = PUT(DCGDIAG,$V0CCS.);

 if addxg = "" then delete;
 proccc1 = put(proc1,$ccsproc.); proccc7 = put(proc7,$ccsproc.);
 proccc2 = put(proc2,$ccsproc.); proccc8 = put(proc8,$ccsproc.);
 proccc3 = put(proc3,$ccsproc.); proccc9 = put(proc9,$ccsproc.);
 proccc4 = put(proc4,$ccsproc.); proccc10 = put(proc10,$ccsproc.);
 proccc5 = put(proc5,$ccsproc.); proccc11 = put(proc11,$ccsproc.);
 proccc6 = put(proc6,$ccsproc.); proccc12 = put(proc12,$ccsproc.);
 proccc13 = put(proc13,$ccsproc.); proccc19 = put(proc19,$ccsproc.);
 proccc14 = put(proc14,$ccsproc.); proccc20 = put(proc20,$ccsproc.);
 proccc15 = put(proc15,$ccsproc.); proccc21 = put(proc21,$ccsproc.);
 proccc16 = put(proc16,$ccsproc.); proccc22 = put(proc22,$ccsproc.);
 proccc17 = put(proc17,$ccsproc.); proccc23 = put(proc23,$ccsproc.);
 proccc18 = put(proc18,$ccsproc.); proccc24 = put(proc24,$ccsproc.);
 proccc25 = put(proc25,$ccsproc.);
************************************************************************************************;
 array ahrqcc{25} $ proccc1 - proccc25;
 * 2/5/2018: include procedure specific PCS surgical category;
 array surg{11} ophtho vascular ortho gen ct uro neuro obgyn plastic ent PCSprocspec;
 array _proc(25) proc1-proc25;

	/*Excluded rehab ccs*/
rehabexcl=0;
if addxg in ('254') then rehabexcl=1;

	/*Excluded PSYCH */
psychexcl = 0;
if addxg in ('650', '651', '652','654', '655', '656', '657', '658', '659',
    '662',  '670') then psychexcl = 1;

	/*Subset Surgeries into catergories */
    *******  Remove CCS 61 from Surgical Cohort   *****;
do i =1 to 25;
    if ahrqcc(i) in ('20', '15', '16', '14', '13', '17') then ophtho=1;
    if ahrqcc(i) in (  '51', '55', '52', '60', '59', '56', '53') then vascular=1;
    if ahrqcc(i) in ('153', '146', '152', '158', '3', '161', '142', '147',
        '162', '148', '154', '145', '150'/*, '151', '143'*/) then ortho=1;
    if ahrqcc(i) in('151', '143') and DCGVRS = "9" then ortho=1;
    if ahrqcc(i) in ('78', '157', '84', '90', '96', '75', '86', '105', '72',
        '80', '73', '85', '164', '74', '167', '176', '89', '166', '99', '94',
        '67', '66', '79') then gen=1;
    if ahrqcc(i) in ('44', '43', '49', '36', '42') then ct=1;
    if ahrqcc(i) in ('101', '112', '103', '104', '114', '118', '113', '106'/*, '109'*/) then uro=1;
    if ahrqcc(i) in ('109') and DCGVRS = "0" then uro=1;
    if ahrqcc(i) in ('1', '9', '2') then neuro=1;
    if ahrqcc(i) in ('172', '175', '28', '144', '160') then plastic=1;
    if ahrqcc(i) in ('33', '10', '12', '26', '21', '23','30','24', '22') then ent=1;
        if ahrqcc(i) in ('124', '119', '132', '129', /*'134','139', '137',*/ '125',
        '131', '120', '123', /*'136', '127', '135',*/ '121', '141', /*'126',*/ '133'/*,
        '122'*/) then obgyn=1;
    if ahrqcc(i) in ('140') and  DCGVRS = "9" then obgyn=1;
    if (ahrqcc(i) in ('32') and _proc(i) in
        ("0C9030Z",
        "0C903ZZ",
        "0C9130Z",
        "0C913ZZ",
        "0C9230Z",
        "0C923ZZ",
        "0C9330Z",
        "0C933ZZ",
        "0C9430Z",
        "0C943ZZ",
        "0C9730Z",
        "0C973ZZ",
        "0C9M30Z",
        "0C9M3ZZ",
        "0C9N30Z",
        "0C9N3ZZ",
        "0C9P30Z",
        "0C9P3ZZ",
        "0C9Q30Z",
        "0C9Q3ZZ",
        "0C9R30Z",
        "0C9R3ZZ",
        "0C9S30Z",
        "0C9S3ZZ",
        "0C9T30Z",
        "0C9T3ZZ",
        "0C9V30Z",
        "0C9V3ZZ",
        "0CPS70Z",
        "0CPS7DZ",
        "0CPS80Z",
        "0CPS8DZ",
        "0W9330Z",
        "0W933ZZ",
        "0W9430Z",
        "0W943ZZ",
        "0W9530Z",
        "0W953ZZ")) or
        (ahrqcc(i) in ('163') and _proc(i) in
        ("0MPX30Z",
        "0MPY30Z",
        "0N9030Z",
        "0N903ZZ",
        "0N9130Z",
        "0N913ZZ",
        "0N9230Z",
        "0N923ZZ",
        "0N9330Z",
        "0N933ZZ",
        "0N9430Z",
        "0N943ZZ",
        "0N9530Z",
        "0N953ZZ",
        "0N9630Z",
        "0N963ZZ",
        "0N9730Z",
        "0N973ZZ",
        "0N9830Z",
        "0N983ZZ",
        "0N9C30Z",
        "0N9C3ZZ",
        "0N9D30Z",
        "0N9D3ZZ",
        "0N9F30Z",
        "0N9F3ZZ",
        "0N9G30Z",
        "0N9G3ZZ",
        "0N9H30Z",
        "0N9H3ZZ",
        "0N9J30Z",
        "0N9J3ZZ",
        "0N9K30Z",
        "0N9K3ZZ",
        "0N9L30Z",
        "0N9L3ZZ",
        "0N9M30Z",
        "0N9M3ZZ",
        "0N9N30Z",
        "0N9N3ZZ",
        "0N9P30Z",
        "0N9P3ZZ",
        "0N9Q30Z",
        "0N9Q3ZZ",
        "0N9X30Z",
        "0N9X3ZZ",
        "0P9030Z",
        "0P903ZZ",
        "0P9130Z",
        "0P913ZZ",
        "0P9230Z",
        "0P923ZZ",
        "0P9330Z",
        "0P933ZZ",
        "0P9430Z",
        "0P943ZZ",
        "0P9530Z",
        "0P953ZZ",
        "0P9630Z",
        "0P963ZZ",
        "0P9730Z",
        "0P973ZZ",
        "0P9830Z",
        "0P983ZZ",
        "0P9930Z",
        "0P993ZZ",
        "0P9B30Z",
        "0P9B3ZZ",
        "0P9C30Z",
        "0P9C3ZZ",
        "0P9D30Z",
        "0P9D3ZZ",
        "0P9F30Z",
        "0P9F3ZZ",
        "0P9G30Z",
        "0P9G3ZZ",
        "0P9H30Z",
        "0P9H3ZZ",
        "0P9J30Z",
        "0P9J3ZZ",
        "0P9K30Z",
        "0P9K3ZZ",
        "0P9L30Z",
        "0P9L3ZZ",
        "0P9M30Z",
        "0P9M3ZZ",
        "0P9N30Z",
        "0P9N3ZZ",
        "0P9P30Z",
        "0P9P3ZZ",
        "0P9Q30Z",
        "0P9Q3ZZ",
        "0P9R30Z",
        "0P9R3ZZ",
        "0P9S30Z",
        "0P9S3ZZ",
        "0P9T30Z",
        "0P9T3ZZ",
        "0P9V30Z",
        "0P9V3ZZ",
        "0PPY30Z",
        "0Q9030Z",
        "0Q903ZZ",
        "0Q9130Z",
        "0Q913ZZ",
        "0Q9230Z",
        "0Q923ZZ",
        "0Q9330Z",
        "0Q933ZZ",
        "0Q9430Z",
        "0Q943ZZ",
        "0Q9530Z",
        "0Q953ZZ",
        "0Q9630Z",
        "0Q963ZZ",
        "0Q9730Z",
        "0Q973ZZ",
        "0Q9830Z",
        "0Q983ZZ",
        "0Q9930Z",
        "0Q993ZZ",
        "0Q9B30Z",
        "0Q9B3ZZ",
        "0Q9C30Z",
        "0Q9C3ZZ",
        "0Q9D30Z",
        "0Q9D3ZZ",
        "0Q9F30Z",
        "0Q9F3ZZ",
        "0Q9G30Z",
        "0Q9G3ZZ",
        "0Q9H30Z",
        "0Q9H3ZZ",
        "0Q9J30Z",
        "0Q9J3ZZ",
        "0Q9K30Z",
        "0Q9K3ZZ",
        "0Q9L30Z",
        "0Q9L3ZZ",
        "0Q9M30Z",
        "0Q9M3ZZ",
        "0Q9N30Z",
        "0Q9N3ZZ",
        "0Q9P30Z",
        "0Q9P3ZZ",
        "0Q9Q30Z",
        "0Q9Q3ZZ",
        "0Q9R30Z",
        "0Q9R3ZZ",
        "0Q9S30Z",
        "0Q9S3ZZ",
        "0QPY30Z"
        )) or
        (ahrqcc(i) in ('174') and _proc(i) in
        ("0W9230Z",
        "0W923ZZ",
        "0W9630Z",
        "0W963ZZ")) or
        (ahrqcc(i) in ('214') and _proc(i) in
        ("0NH005Z",
        "0NH035Z",
        "0NH045Z")) then
        PCSprocspec = 1; 
            
end;

do j=1 to 11;
    if surg(j)=. then surg(j)=0;
end;
surg_sum=sum(ophtho, vascular, ortho, gen, ct, uro, neuro, obgyn, plastic, ent, PCSprocspec);
attrib category length=$10.;


if ophtho or vascular or ortho or gen or ct or uro or neuro or plastic or ent or obgyn or PCSprocspec then
    category='Surgical';
else category='Medical';

attrib subcategory length=$18.;

if addxg in ('11', '12', '13', '14', '15', '16', '17', '18', '19', '20', '21', '22',
    '23', '24', '29', '30', '31', '32', '33', '34', '35', '36', '37', '38', '39', '40',
    '41', '42', '43', '44', '45','25','26','27','28') then subcategory='Cancer';
else if addxg in ('56','103', '108', '122', '125', '127', '128', '131')
                                then subcategory='Cardiorespiratory';
else if addxg in ('96', '97', '100', '101', '102', '104', '105',
    '106', '107', '114', '115', '116', '117', '213') then subcategory='CV';
else if addxg in ('1', '2', '3', '4', '5', '6', '7', '8', '9', '10', '47', '48', '49',
    '50', '51', '52', '53',    '54', '55', '57', '58', '59', '60', '61', '62', '63',
    '64', '76', '77', '84', '86', '87', '88', '89',    '90', '91', '92', '93', '94', '98',
    '99', '118', '119', '120', '121', '123', '124', '126', '129',
    '130', '132', '133', '134', '135', '136', '137', '138', '139', '140', '141', '142',
    '143', '144', '145', '146', '147', '148', '149', '151', '152', '153', '154', '155',
    '156', '157', '158', '159', '160', '161', '162', '163', '164', '165', '166', '167',
    '173', '175', '197', '198', '199', '200', '201', '202',    '203', '204', '205', '206',
    '207', '208', '209', '210', '211', '212', '214', '215', '217', '225', '226',
    '228', '229','230', '231', '232', '234', '235', '236', '237', '238', '239', '240',
    '242', '243', '244', '245', '246', '247', '248', '249', '250', '251', '252', '253',
    '255', '256', '257', '258', '259',  '241', '168','170','172','46','171','169',/*'174',*/
    '653','661','660','663' )
    or (addxg in ('2617') and DCGVRS = "0")
        then subcategory='Medicine';
else if addxg in ('78', '79', '80', '81', '82', '83', '85', '95', '109', '110', '111',
    '112', '113', '216', '227', '233') then subcategory='Neurology';

*else if addxg in ('176','177','178','179','180','181','182','183','184','185','186','187',
'188','189','190','191','192','193','194','195','196') then subcategory='OB';
    ***** OB is not part of the Medicare measure - these codes added for all-payer*****;

if category='Surgical' then subcategory='Surgical';
if rehabexcl=1 then subcategory =('Rehab excl');
if psychexcl=1 then subcategory =('Psych excl');

/*Excluded PPS-Exempt cancer hopitals*/
if provider in ('050146','050660','100079','100271','220162','330354','360242','390196','450076','330154','500138')
then cancer_hosp=1;
else cancer_hosp = 0;

all = 1;

/* TRANSFER OUT INDICATOR, SAME PTS, DIFFERENT HOSP, 0 OR 1 DAY IN BETWEEN, NOT V CODE VISIT */
TRANSFER_OUT=(ADMIT <= LAG(ADMIT) <= DISCH +1) AND (bid_mpr_3=LAG(bid_mpr_3)) AND (PROVIDER ^=LAG(PROVIDER));
TRANS_COMBINE=(TRANSFER_OUT OR TRANS_FIRST OR TRANS_MID or post_flag);

MALE=(SEX='1');
AGE=INT((ADMIT - BIRTH)/365.25);
AGE65=(AGE >=65);
AGE_65=AGE - 65;

DEAD=(DDEST=20);
AMA=(DDEST=7);

IF DEATH ^=. THEN DO;
    IF 0 < (DEATH - DISCH) <=30 THEN DD30=1;
    ELSE IF (DEATH - DISCH) > 30 THEN DD30=0;
    ELSE IF (DEATH - DISCH) <= 0 THEN DD30=0;
    END;
ELSE DD30=0;
obs30_mort=dd30;

/*PREMO_A is PART A enrollment only*/
PRIOR12=(PREMO_A=12);   

*If pt dies in 30day period they are eligible;
IF DD30=1 THEN POSTMOD_A=1;   
POST1=(POSTMOD_A IN (1, 2, 3));


/* INCLUSION CRITERIA: FFS,AGE GE 65,IN HOSPITAL DEATH,WITHOUT >= 1 MOTNH POST,
   EXCLUSION  CRITERIA: TRANSFER OUT, WITH 12-MONTH HISTORY,PPS Cancer Hosp, Rehab,
                        CANCER Medical,AMA, psych removed pre-dryrun-PUT IN AGAIN */
if dead=0 and age65 and post1=1 and trans_combine=0 and PRIOR12=1 and ama = 0 and
  rehabexcl=0 and cancer_hosp = 0  and subcategory not in ('Cancer') and psychexcl=0
then sample = 1; else sample=0 ;
RUN;

PROC SORT DATA=ALL;
BY bid_mpr_3 CASE_P;
RUN;

/*Define Planned Readmissions Version 4.0 in Post Index File*/
%Post_planned(post=postindex);

proc sort data=postindex;
    by bid_mpr_3 case admit;
run;

data readm1 QA_DupIndex;
merge ALL (IN=A)
    postindex (IN=B KEEP=bid_mpr_3 ADMIT DISCH PROVIDER DIAG1 CLM_DGNS_PRCDR_ICD_IND PROCCCP_1 - PROCCCP_25 CASE planned    procnum proc1-proc25
                RENAME=(DIAG1=_DIAG1 CLM_DGNS_PRCDR_ICD_IND  = _CLM_DGNS_PRCDR_ICD_IND ADMIT=_ADMIT DISCH=_DISCH PROVIDER=_PROVIDER proc1=_proc1 proc2=_proc2
                       proc3=_proc3    proc4=_proc4    proc5=_proc5    proc6=_proc6    proc7=_proc7    proc8=_proc8
                       proc9=_proc9    proc10=_proc10    proc11=_proc11    proc12=_proc12    proc13=_proc13    proc14=_proc14
                       proc15=_proc15    proc16=_proc16    proc17=_proc17    proc18=_proc18    proc19=_proc19    proc20=_proc20
                       proc21=_proc21    proc22=_proc22    proc23=_proc23    proc24=_proc24    proc25=_proc25     CASE=CASE_P  ));
by bid_mpr_3 CASE_P;
IF A;

/* RADM30ALL: ANY READMISSION WITHIN 30 DAYS */
IF NOT B THEN RADM30ALL=0;
ELSE IF 0 <= _ADMIT - DISCH <=30 THEN RADM30ALL=1;
ELSE IF _ADMIT - DISCH > 30 THEN RADM30ALL=0;

interval = .;
same = .;
if b then do;
    INTERVAL=_ADMIT - DISCH;
    SAME=(PROVIDER=_PROVIDER);
    end;
RADM30=RADM30ALL;

    radm30p=0;
    if planned =1 and RADM30=1 then do;
        RADM30 = 0;
        radm30p = 1;
        end;

    if _CLM_DGNS_PRCDR_ICD_IND = '9' then DO;
    /*Any readmission with principal diagnosis eq V57  is not counted as readmission*/
    if upcase(_diag1)=:'V57'  then Radm_rehab=1;
    else Radm_rehab=0;

    /*Any readmission with psych principal diagnosis eq in range of 290-319 that was
        within 1 day of the discharge date of index admission with discharge dispostion
        eq 65 is not counted as readmission*/
    if (_diag1=:'29' or _diag1=:'30' or _diag1=:'31') and (interval in (0,1))  and
        ddest=65 then Radm_psy=1; else Radm_psy=0;
    END;

if _CLM_DGNS_PRCDR_ICD_IND = '0' then DO;
    /*Any readmission with in day 0 or 1 with discharge disp = 62 is not counted as readmission*/
if (interval in (0,1))  and ddest=62 then Radm_rehab=1;
else Radm_rehab=0;

    /*Any readmission with psych principal diagnosis
        within 1 day of the discharge date of index admission with discharge dispostion
        eq 65 is not counted as readmission*/
if (_diag1=:'F' ) and (interval in (0,1))  and
    ddest=65 then Radm_psy=1; else Radm_psy=0;
END;

/*Will not include any rehab or possible psych transfers in our planned readmission logic  */
if radm_rehab=1 and (radm30=1 or radm30p = 1) then do; radm30=0; radm30p = 0; interval = 999;  end;
if radm_psy=1 and (radm30=1 or radm30p = 1) then do; radm30=0; radm30p = 0; interval = 999;     end;

bid_mpr_3_case=strip(bid_mpr_3)||'_'||strip(put(case_p,3.));

/* PART OF TRANS BUNDLE, SHOULD HAVE BEEN EXCLUDED *
IF RADM30=1 AND RADM_AMI=1 AND INTERVAL=0 AND SAME=1 THEN TRANS_BUNDLE=1;
ELSE TRANS_BUNDLE=0;
IF TRANS_BUNDLE=1 THEN SAMPLE=0; */

*** Same day bundle index should be excluded;
If RADM30=1 and interval=0 and same=1 and diag1=_diag1 then do;
    Bundle=1;
    Sample=0;
    end;

DROP I;
IF ADMIT=_ADMIT AND DISCH=_DISCH AND PROVIDER=_PROVIDER AND DIAG1=_DIAG1 THEN OUTPUT QA_DupIndex;
ELSE OUTPUT readm1;
run;

proc sort data=readm1;
*ICD-10: Added another level to account for transfers in the postindex file;
    by bid_mpr_3_case interval descending _disch;
run;

data readm1data;
set readm1;
by bid_mpr_3_case;
if first.bid_mpr_3_case;
DROP bid_mpr_3_CASE;
run;

proc sort data=readm1data;
by bid_mpr_3 case;
run;

DATA sample;
set readm1data (where=(sample=1));
csex=put(sex, $1.);
attrib addxgnum length=8.; 
addxgnum=input(addxg, 8.);
drop sex;
RUN;

* DETERMINE GRAB BAG CATEGORIES;
proc sql noprint;
create table temp_indicator as
select distinct subcategory, addxg, count(bid_mpr_3) as addxgvol
from sample
group by subcategory, addxg
having addxgvol < 1000; 
quit;

proc sql noprint;
select unique(addxg)
into:  med_lfaddxg separated by ','
from temp_indicator
where subcategory='Medicine';

select unique(addxg)
into:  surg_lfaddxg separated by ','
from temp_indicator
where subcategory='Surgical';

select unique(addxg)
into:  cardio_lfaddxg separated by ','
from temp_indicator
where subcategory='Cardiorespiratory';

select unique(addxg)
into:  cv_lfaddxg separated by ','
from temp_indicator
where subcategory='CV';

select unique(addxg)
into:  neuro_lfaddxg separated by ','
from temp_indicator
where subcategory='Neurology';

quit;

DATA _NULL_;

%LET MFILLER=0;

IF SYMEXIST('MED_LFADDXG')=0 THEN DO;
    CALL SYMPUT('MED_LFADDXG', '&MFILLER');
    END;
IF SYMEXIST('SURG_LFADDXG')=0 THEN DO;
    CALL SYMPUT('SURG_LFADDXG', '&MFILLER');
    END;
IF SYMEXIST('CV_LFADDXG')=0 THEN DO;
    CALL SYMPUT('CV_LFADDXG', '&MFILLER');
    END;
IF SYMEXIST('CARDIO_LFADDXG')=0 THEN DO;
    CALL SYMPUT('CARDIO_LFADDXG', '&MFILLER');
    END;
IF SYMEXIST('NEURO_LFADDXG')=0 THEN DO;
    CALL SYMPUT('NEURO_LFADDXG', '&MFILLER');
    END;

RUN;

****************************************************************;
*Create risk factors for adjustment                            *;
****************************************************************;

DATA HXDIAG;
SET &HXDIAG &TRANS_DIAG;
if source in ('0.0.1.0','0.0.2.0'); /* only ip history available */
RUN;



PROC SQL;
CREATE TABLE HXDIAG2 AS
SELECT SAMPLE.bid_mpr_3, SAMPLE.CASE, SAMPLE.AGE, SAMPLE.CSEX AS SEX,
    HXDIAG.DIAG,
    HXDIAG.vrsn,
    HXDIAG.FDATE,
    HXDIAG.SOURCE,
    HXDIAG.TDATE, HXDIAG.YEAR
FROM SAMPLE LEFT JOIN HXDIAG
    ON SAMPLE.bid_mpr_3=HXDIAG.bid_mpr_3 AND SAMPLE.HISTORY_CASE=HXDIAG.CASE;
QUIT;


%HCCPAI(sample, &CONDITION._PA0);

data  &CONDITION._PA2 &CONDITION._PA1;
set hxdiag2 (keep=bid_mpr_3 case AGE SEX diag vrsn source where=(diag^=''));
attrib icd length=$7.;
attrib vrs length=$1.;
icd=diag;
vrs=vrsn;
if source in ('0.0.1.0') then output &CONDITION._PA1;
else if source in ('0.0.2.0') then output &CONDITION._PA2;
Run;

%cms_hcc_get(&CONDITION._PA0, &CONDITION._PA0_CC, PA0);

%cms_hcc_get(&CONDITION._PA1, &CONDITION._PA1_CC, PA1);

%cms_hcc_get(&CONDITION._PA2, &CONDITION._PA2_CC, PA2);

PROC SORT DATA=&CONDITION._PA0_CC;
    BY bid_mpr_3 CASE;
RUN;
PROC SORT DATA=&CONDITION._PA2_CC;
    BY bid_mpr_3 CASE;
RUN;
PROC SORT DATA=&CONDITION._PA1_CC;
    BY bid_mpr_3 CASE;
RUN;

data &ANALYSIS;
merge     sample (in=a rename = (_ADMIT=ADMIT_POST _DISCH=DISCH_POST CSEX=SEX))
        &CONDITION._PA0_CC (rename=(nhic=nhic0))
        &CONDITION._PA1_CC (rename=(nhic=nhic1))
        &CONDITION._PA2_CC (rename=(nhic=nhic2))
                ;
by bid_mpr_3 case;
if a;

ARRAY PA0{1:201} PA0CC1-PA0CC201;
ARRAY PA2{1:201} PA2CC1-PA2CC201;
ARRAY PA1{1:201} PA1CC1-PA1CC201;
ARRAY CC{1:201} CC1 - CC201;

/* HF readmission approach for Complication Derivation : UPDATED WITH v22 11/17/16  BY YALE*/
* 12/14/2017: non-PRA update;
    ARRAY COMP{*}    PA0CC2   PA0CC7   PA0CC17  PA0CC24
        /*PA0CC30 PA0CC33*/ PA0CC36
        PA0CC48  /*PA0CC50  PA0CC80*/  PA0CC82  /*PA0CC83*/ PA0CC84 PA0CC85
        PA0CC86  PA0CC87  PA0CC96  PA0CC97
        /*PA0CC99*/  /*PA0CC100 PA0CC101*/ PA0CC103 PA0CC104
        PA0CC106 PA0CC107 PA0CC108 PA0CC109 PA0CC114 PA0CC115
        /*PA0CC117*/ PA0CC134 PA0CC135 PA0CC140 /*PA0CC141 PA0CC142 PA0CC144*/
        PA0CC157 PA0CC158 PA0CC159 PA0CC160
        /*PA0CC164 PA0CC166 PA0CC167 PA0CC168*/ PA0CC170
        /*PA0CC171 PA0CC173 PA0CC175 PA0CC176 PA0CC177*/
        PA0CC186 /*PA0CC187 PA0CC188*/ PA0CC189 PA0CC190 /*PA0CC191*/
        ;

    DO I=1 TO dim(COMP);
        IF COMP(I)=1 THEN COMP(I)=0;
        END;


DO I=1 TO 201;
    IF PA0[I]=. THEN PA0[I]=0;
    IF PA2[I]=. THEN PA2[I]=0;
    IF PA1[I]=. THEN PA1[I]=0;
    CC[I]=PA0[I] OR PA2[I] OR PA1[I];
END;
Attrib Cohort length=$18.;
IF subcategory='Cardiorespiratory'  THEN Cohort='CARDIORESPIRATORY';
ELSE IF subcategory='CV' THEN Cohort='CV';
ELSE IF subcategory='Neurology' THEN Cohort='NEUROLOGY';
ELSE IF subcategory='Medicine' THEN Cohort='MEDICINE';
ELSE IF  subcategory='Surgical' then Cohort='SURGICAL';
%ACR_model_variables();
%ACR_model_Condition_Indicator();

If Cohort in ('SURGICAL', 'MEDICINE', 'CV', 'CARDIORESPIRATORY', 'NEUROLOGY');
KEEP &MODEL_VAR bid_mpr_3 CASE ADMIT DISCH provider COHORT CATEGORY condition radm30
     radm30p obs30_mort
     age dd30 radm30all planned trans_combine age65 
     tin_assigned stayid
     ;
label
   CONDITION = 'Condition code'
   Cohort = 'Condition subcategory'
   RADM30 = '30-day readmission flag'
   admit = 'Admission date'
   category = 'Condition category'
   disch = 'Discharge date'
   obs30_mort = 'Died within 30 days of discharge flag'
   radm30p = 'Planned 30-day readmission flag'
   radm30all = 'All 30-day readmission flag'
   planned = 'Planned procedure flag'
;
RUN;



data &all;
merge readm1data (in=a)
    &analysis (keep=bid_mpr_3 case &model_var cohort condition
               tin_assigned stayid
               rename = (age_65 = age65_2
                         stayid = stayid_2
                         tin_assigned = tin_assigned2)
               ) ;
by bid_mpr_3 case;
if a;
diffage = (age_65 ne age65_2) ;
diffstay = (stayid ne stayid_2) ;
diff_tin = (tin_assigned ne tin_assigned2) ;
run;

*************Run HGLM Analyses by Cohort *******************;
* TIN replaces ACO, mod_z option;
%HGLM_CONDITION_TIN_MOD_Z(&ANALYSIS, CARDIORESPIRATORY);
%HGLM_CONDITION_TIN_MOD_Z(&ANALYSIS, CV);
%HGLM_CONDITION_TIN_MOD_Z(&ANALYSIS, NEUROLOGY);
%HGLM_CONDITION_TIN_MOD_Z(&ANALYSIS, MEDICINE);
%HGLM_CONDITION_TIN_MOD_Z(&ANALYSIS, SURGICAL);


*********************************************************************************;
*****  Calculate RSRR by Cohort and Composite************************************;
*********************************************************************************;

title;
PROC SQL;
CREATE TABLE tin_assigned AS
SELECT DISTINCT tin_assigned,
        COUNT(tin_assigned) AS VOLUME,
        SUM(RADM30) AS READMISSION,
        SUM(RADM30p) AS READM_PLAN
FROM &ANALYSIS
GROUP BY tin_assigned;
SELECT MEAN(RADM30) INTO: HWYBAR FROM &ANALYSIS;
QUIT;


DATA &ACR_RSRR;
MERGE   tin_assigned (IN=A)
        MEDICINE_RSRR (IN=B KEEP=tin_assigned SRR VOLUME RSRR OBS RADM RADMP
                        RENAME=(SRR=SRR_MED VOLUME=VOLUME_MED RSRR=RSRR_MED OBS=OBS_MED
        RADM=RADM30_MED RADMP=RADM30P_MED))
        SURGICAL_RSRR (IN=C KEEP=tin_assigned SRR VOLUME RSRR OBS RADM RADMP
                        RENAME=(SRR=SRR_SURG VOLUME=VOLUME_SURG RSRR=RSRR_SURG OBS=OBS_SURG
        RADM=RADM30_SURG        RADMP=RADM30P_SURG))
                CV_RSRR (IN=E KEEP=tin_assigned SRR VOLUME RSRR OBS RADM RADMP
                        RENAME=(SRR=SRR_CV VOLUME=VOLUME_CV RSRR=RSRR_CV OBS=OBS_CV
        RADM=RADM30_CV RADMP=RADM30P_CV))
                 CARDIORESPIRATORY_RSRR  (IN=F KEEP=tin_assigned SRR VOLUME RSRR OBS RADM RADMP
                        RENAME=(SRR=SRR_CARDIO VOLUME=VOLUME_CARDIO RSRR=RSRR_CARDIO OBS=OBS_CARDIO
        RADM=RADM30_CARDIO RADMP=RADM30P_CARDIO))
                NEUROLOGY_RSRR (IN=H KEEP=tin_assigned SRR VOLUME RSRR OBS RADM RADMP
                        RENAME=(SRR=SRR_NEURO VOLUME=VOLUME_NEURO RSRR=RSRR_NEURO OBS=OBS_NEURO
        RADM=RADM30_NEURO RADMP=RADM30P_NEURO));
BY tin_assigned;

IF A;


ARRAY RESET0{5}  VOLUME_MED VOLUME_SURG VOLUME_CV VOLUME_CARDIO VOLUME_NEURO;
DO I=1 TO 5;
        IF RESET0(I)=. THEN RESET0(I)=0;
END;

/* 1 threshold indicator */
MED_VOL1=(VOLUME_MED >=1); 
SUR_VOL1=(VOLUME_SURG >=1);
CV_VOL1=(VOLUME_CV >=1);
CAR_VOL1=(VOLUME_CARDIO >=1);
NEU_VOL1=(VOLUME_NEURO >=1);

/* 25 threshold indicator */
MED_VOL25=(VOLUME_MED >=25); 
SUR_VOL25=(VOLUME_SURG >=25);
CV_VOL25=(VOLUME_CV >=25);
CAR_VOL25=(VOLUME_CARDIO >=25);
NEU_VOL25=(VOLUME_NEURO >=25);

/* # of cohorts with at least 1 cases */
NUMBER_COHORT1=SUM(MED_VOL1, SUR_VOL1, CV_VOL1, CAR_VOL1, NEU_VOL1);

/* # of cohorts with at least 25 cases */
NUMBER_COHORT25=SUM(MED_VOL25, SUR_VOL25, CV_VOL25, CAR_VOL25, NEU_VOL25);

IF VOLUME_MED>0 THEN MED_NUM=VOLUME_MED*LOG(SRR_MED); ELSE MED_NUM=0;
IF VOLUME_SURG>0 THEN SURG_NUM=VOLUME_SURG*LOG(SRR_SURG); ELSE SURG_NUM=0;
IF VOLUME_CV>0 THEN CV_NUM=VOLUME_CV*LOG(SRR_CV); ELSE CV_NUM=0;
IF VOLUME_CARDIO>0 THEN CARDIO_NUM=VOLUME_CARDIO*LOG(SRR_CARDIO); ELSE CARDIO_NUM=0;
IF VOLUME_NEURO>0 THEN NEURO_NUM=VOLUME_NEURO*LOG(SRR_NEURO); ELSE NEURO_NUM=0;

TOTAL_NUM=SUM(MED_NUM, SURG_NUM, CV_NUM, CARDIO_NUM, NEURO_NUM);

SRR_ACR=EXP(TOTAL_NUM/VOLUME);
RSRR_ACR=(SRR_ACR*&HWYBAR)*100;
OBS_ACR=(READMISSION/VOLUME)*100;

/*** ADD LABELS ***/
ATTRIB
SRR_MED LABEL='STANDARDIZED RISK RATIO, MEDICAL' LENGTH=8.
RSRR_MED LABEL='RISK-STANDARDIZED READMISSION RATE, MEDICAL' LENGTH=8.
OBS_MED LABEL='RAW READMISSION RATE, MEDICAL' LENGTH=8.
RADM30_MED LABEL='NUMBER OF 30-DAY READMISSIONS, MEDICAL' LENGTH=8.
RADM30P_MED LABEL='NUMBER OF PLANNED 30-DAY READMISSIONS, MEDICAL' LENGTH=8.

SRR_SURG LABEL='STANDARDIZED RISK RATIO, SURGICAL' LENGTH=8.
RSRR_SURG LABEL='RISK-STANDARDIZED READMISSION RATE, SURGICAL' LENGTH=8.
OBS_SURG LABEL='RAW READMISSION RATE, SURGICAL' LENGTH=8.
RADM30_SURG LABEL='NUMBER OF 30-DAY READMISSIONS, SURGICAL' LENGTH=8.
RADM30P_SURG LABEL='NUMBER OF PLANNED 30-DAY READMISSIONS, SURGICAL' LENGTH=8.

SRR_CV LABEL='STANDARDIZED RISK RATIO, CV' LENGTH=8.
RSRR_CV LABEL='RISK-STANDARDIZED READMISSION RATE, CV' LENGTH=8.
OBS_CV LABEL='RAW READMISSION RATE, CV' LENGTH=8.
RADM30_CV LABEL='NUMBER OF 30-DAY READMISSIONS, CV' LENGTH=8.
RADM30P_CV LABEL='NUMBER OF PLANNED 30-DAY READMISSIONS, CV' LENGTH=8.

SRR_CARDIO LABEL='STANDARDIZED RISK RATIO, CARDIORESPIRATORY' LENGTH=8.
RSRR_CARDIO LABEL='RISK-STANDARDIZED READMISSION RATE, CARDIORESPIRATORY' LENGTH=8.
OBS_CARDIO LABEL='RAW READMISSION RATE, CARDIORESPIRATORY' LENGTH=8.
RADM30_CARDIO LABEL='NUMBER OF 30-DAY READMISSIONS, CARDIORESPIRATORY' LENGTH=8.
RADM30P_CARDIO LABEL='NUMBER OF PLANNED 30-DAY READMISSIONS, CARDIORESPIRATORY' LENGTH=8.

SRR_NEURO LABEL='STANDARDIZED RISK RATIO, NEUROLOGY' LENGTH=8.
RSRR_NEURO LABEL='RISK-STANDARDIZED READMISSION RATE, NEUROLOGY' LENGTH=8.
OBS_NEURO LABEL='RAW READMISSION RATE, NEUROLOGY' LENGTH=8.
RADM30_NEURO LABEL='NUMBER OF 30-DAY READMISSIONS, NEUROLOGY' LENGTH=8.
RADM30P_NEURO LABEL='NUMBER OF PLANNED 30-DAY READMISSIONS, NEUROLOGY' LENGTH=8.

MED_VOL1 LABEL='THRESHOLD IND OF ELIG ADMISSIONS >= 1, MEDICAL' LENGTH=3.
SUR_VOL1 LABEL='THRESHOLD IND OF ELIG ADMISSIONS >= 1, SURGICAL' LENGTH=3.
CV_VOL1  LABEL='THRESHOLD IND OF ELIG ADMISSIONS >= 1, CV' LENGTH=3.
CAR_VOL1 LABEL='THRESHOLD IND OF ELIG ADMISSIONS >= 1, CARDIORESPIRATORY' LENGTH=3.
NEU_VOL1 LABEL='THRESHOLD IND OF ELIG ADMISSIONS >= 1, NEUROLOGY' LENGTH=3.
NUMBER_COHORT1 LABEL='NBR OF COHORTS WITH ELIG ADMISSIONS >= 1' LENGTH=3.

MED_VOL25 LABEL='THRESHOLD IND OF ELIG ADMISSIONS >= 25, MEDICAL' LENGTH=3.
SUR_VOL25 LABEL='THRESHOLD IND OF ELIG ADMISSIONS >= 25, SURGICAL' LENGTH=3.
CV_VOL25  LABEL='THRESHOLD IND OF ELIG ADMISSIONS >= 25, CV' LENGTH=3.
CAR_VOL25 LABEL='THRESHOLD IND OF ELIG ADMISSIONS >= 25, CARDIORESPIRATORY' LENGTH=3.
NEU_VOL25 LABEL='THRESHOLD IND OF ELIG ADMISSIONS >= 25, NEUROLOGY' LENGTH=3.
NUMBER_COHORT25 LABEL='NBR OF COHORTS WITH ELIG ADMISSIONS >= 25' LENGTH=3.

VOLUME LABEL='NUMBER OF ELIGIBLE ADMISSIONS, OVERALL' LENGTH=8.
READMISSION LABEL='NUMBER OF READMISSIONS, OVERALL' LENGTH=8.
READM_PLAN LABEL='NUMBER OF PLANNED READMISSIONS, OVERALL' LENGTH=8.

VOLUME_MED LABEL='NUMBER OF ELIGIBLE ADMISSIONS, MEDICAL' LENGTH=8.
VOLUME_SURG LABEL='NUMBER OF ELIGIBLE ADMISSIONS, SURGICAL' LENGTH=8.
VOLUME_CV LABEL='NUMBER OF ELIGIBLE ADMISSIONS, CV' LENGTH=8.
VOLUME_CARDIO LABEL='NUMBER OF ELIGIBLE ADMISSIONS, CARDIORESPIRATORY' LENGTH=8.
VOLUME_NEURO LABEL='NUMBER OF ELIGIBLE ADMISSIONS, NEUROLOGY' LENGTH=8.
MED_NUM LABEL='ADJUSTED NUMBER OF ELIGIBLE ADMISSIONS, MEDICAL' LENGTH=8.
SURG_NUM LABEL='ADJUSTED NUMBER OF ELIGIBLE ADMISSIONS, SURGICAL' LENGTH=8.
CV_NUM LABEL='ADJUSTED NUMBER OF ELIGIBLE ADMISSIONS, CV' LENGTH=8.
CARDIO_NUM LABEL='ADJUSTED NUMBER OF ELIGIBLE ADMISSIONS, CARDIORESPIRATORY' LENGTH=8.
NEURO_NUM LABEL='ADJUSTED NUMBER OF ELIGIBLE ADMISSIONS, NEUROLOGY' LENGTH=8.

TOTAL_NUM LABEL='ADJUSTED NUMBER OF ELIGIBLE ADMISSIONS, OVERALL' LENGTH=8.
SRR_ACR LABEL='STANDARDIZED RISK RATIO, OVERALL' LENGTH=8.
RSRR_ACR LABEL='RISK-STANDARDIZED READMISSION RATE, OVERALL' LENGTH=8.
OBS_ACR LABEL='RAW READMISSION RATE, OVERALL' LENGTH=8.
;

drop i;

RUN;

proc datasets lib = work memtype = data kill ;
quit ;
run ;

*******************************************************************************************;
* Bootstrap: Need to specify output file, starting point, end point and seed              *;
*******************************************************************************************;
/*
%LET START=  1;
%LET NUMBER= ;
%LET RSEED=  10000;
%let nbrits     = 100;
%let firstmethod= NRRIDG;
%let nextmethod = NMSIMP;


%BOOTSTRAP_HOSPITAL_READMISSION(&ANALYSIS, &RESULTS, &START, &NUMBER, &RSEED);
*/

